<?php
/* --------------------------------------------------------------
   ParcelServiceDetails.php 2020-03-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\Model;

use Webmozart\Assert\Assert;

/**
 * Class ParcelServiceDetails
 *
 * @package Gambio\Admin\TrackingCode\Model
 */
class ParcelServiceDetails implements \JsonSerializable
{
    /**
     * @var int
     */
    private $id;
    
    /**
     * @var string
     */
    private $name;
    
    /**
     * @var string
     */
    private $url;
    
    /**
     * @var string
     */
    private $comment;
    
    
    /**
     * ParcelServiceDetails constructor.
     *
     * @param int    $id
     * @param string $name
     * @param string $url
     * @param string $comment
     */
    private function __construct(int $id, string $name, string $url, string $comment)
    {
        $this->id      = $id;
        $this->name    = $name;
        $this->url     = $url;
        $this->comment = $comment;
    }
    
    
    /**
     * @param int    $id
     * @param string $name
     * @param string $url
     * @param string $comment
     *
     * @return ParcelServiceDetails
     */
    public static function create(int $id, string $name, string $url, string $comment): ParcelServiceDetails
    {
        Assert::greaterThan($id, 0, 'Invalid parcel service ID provided. Got: %s');
        Assert::notWhitespaceOnly($name, 'Parcel service name can not be empty.');
        
        return new self($id, $name, $url, $comment);
    }
    
    
    /**
     * @return int
     */
    public function id(): int
    {
        return $this->id;
    }
    
    
    /**
     * @return string
     */
    public function name(): string
    {
        return $this->name;
    }
    
    
    /**
     * @return string
     */
    public function url(): string
    {
        return $this->url;
    }
    
    
    /**
     * @return string
     */
    public function comment(): string
    {
        return $this->comment;
    }
    
    
    /**
     * @inheritDoc
     */
    public function jsonSerialize(): array
    {
        return [
            'id'      => $this->id(),
            'name'    => $this->name(),
            'url'     => $this->url(),
            'comment' => $this->comment(),
        ];
    }
    
    
    /**
     * @param array $data
     *
     * @return ParcelServiceDetails
     */
    public static function createFromArray(array $data): ParcelServiceDetails
    {
        return self::create($data['id'], $data['name'], $data['url'], $data['comment']);
    }
}